"""
build_lattice.py
~~~~~~~~~~~~~~~~~

Construct a two‑dimensional square lattice with periodic boundary conditions.

The lattice is represented as a NumPy array of length ``size × size × 2`` of
tuples ``((x, y), μ)`` where ``μ ∈ {0, 1}`` labels the two coordinate
directions.  A link ``((x, y), 0)`` connects the site at ``(x, y)`` to the
site at ``(x+1, y)`` (modulo the lattice size), while ``((x, y), 1)``
connects ``(x, y)`` to ``(x, y+1)``.  Periodic boundaries ensure that every
site has exactly two outgoing links.

When run as a script ``main`` reads a configuration YAML file, extracts the
``lattice_size`` and ``data_dir`` parameters and writes the lattice to
``data_dir/lattice.npy``.  The lattice is always rebuilt so that changing
``lattice_size`` in the configuration takes effect on subsequent runs.
"""

from __future__ import annotations

import os
import yaml
import numpy as np
from typing import List, Tuple


def build_lattice(size: int = 4, boundary: str = 'periodic') -> np.ndarray:
    """Construct a square lattice as a list of links.

    Parameters
    ----------
    size : int
        Number of sites along each dimension of the square lattice.
    boundary : str
        Either ``'periodic'`` (wrap around at boundaries) or ``'open'``
        (exclude links that would exit the lattice).  Only periodic
        boundaries are used in the Volume 4 pipeline.

    Returns
    -------
    numpy.ndarray
        Array of shape ``(size × size × 2,)`` where each element is a
        ``((x, y), μ)`` tuple describing a link.
    """
    links: List[Tuple[Tuple[int, int], int]] = []
    # μ=0 corresponds to the +x direction, μ=1 to the +y direction
    directions = [(1, 0), (0, 1)]
    for x in range(size):
        for y in range(size):
            for mu, (dx, dy) in enumerate(directions):
                nx = x + dx
                ny = y + dy
                if boundary == 'periodic':
                    nx %= size
                    ny %= size
                else:
                    # skip links that would leave the lattice
                    if not (0 <= nx < size and 0 <= ny < size):
                        continue
                links.append(((x, y), mu))
    return np.array(links, dtype=object)


def main(config_path: str = 'config.yaml') -> None:
    """Entry point for building and writing the lattice to disk.

    The configuration file should contain at least ``lattice_size`` and
    optionally a ``data_dir`` and ``lattice_file``.  If ``data_dir`` is
    relative it is interpreted relative to the directory containing the
    configuration file.  The lattice is always written to disk, even if
    it already exists, to ensure that changing the configuration has an
    immediate effect.
    """
    # Resolve configuration path – allow relative path relative to CWD
    candidate = config_path if os.path.isabs(config_path) else os.path.abspath(config_path)
    if not os.path.exists(candidate):
        raise FileNotFoundError(f"Configuration file not found: {config_path}")
    with open(candidate) as f:
        cfg = yaml.safe_load(f)
    lattice_size = int(cfg.get('lattice_size', 4))
    # Determine where to write the lattice
    base_dir = os.path.dirname(candidate)
    data_dir_cfg = cfg.get('data_dir', 'data')
    if os.path.isabs(data_dir_cfg):
        data_dir = data_dir_cfg
    else:
        data_dir = os.path.join(base_dir, data_dir_cfg)
    os.makedirs(data_dir, exist_ok=True)
    lattice_file = cfg.get('lattice_file', 'lattice.npy')
    lattice_path = os.path.join(data_dir, lattice_file)
    # Build and save the lattice
    lattice = build_lattice(size=lattice_size, boundary='periodic')
    np.save(lattice_path, lattice)
    return None


if __name__ == '__main__':
    import sys
    cfg_path = sys.argv[1] if len(sys.argv) > 1 else 'config.yaml'
    main(cfg_path)